/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : AES-GCM(128bit) implementation.
 *--------------------------------------------------------------------
 * MODULE        :
 * DEPENDENCIES  :
 ********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>

#define NONCE_LEN 12        /* AES use 12 bytes of nonce, then extend to 16 bytes IV( initial vector ) */
#define AES_BLOCK_SIZE 16    /* AES block size is 16 bytes */
#define EXTKEY_LEN 44        /* AES-128 extends key to 44 bytes */

#ifdef __linux__
/* for Linux */
#if (defined(__BigEndian) || defined(BigEndian) || (defined(BYTE_ORDER) && defined(BIG_ENDIAN) && (BYTE_ORDER == BIG_ENDIAN)))
#define cmChangeByteOrder32(_v)    ((((_v) & 0xFF000000) >> 24) |  \
                                    (((_v) & 0x00FF0000) >> 8)  |  \
                                    (((_v) & 0x0000FF00) << 8)  |  \
                                    (((_v) & 0x000000FF) << 24)    \
                                   )
#else /* (defined(__BigEndian) || defined(BigEndian) || (defined(BYTE_ORDER) && defined(BIG_ENDIAN) && (BYTE_ORDER == BIG_ENDIAN))) */
#define cmChangeByteOrder32(_v)    (_v)
#endif /* (defined(__BigEndian) || defined(BigEndian) || (defined(BYTE_ORDER) && defined(BIG_ENDIAN) && (BYTE_ORDER == BIG_ENDIAN))) */
#else /* __linux__ */
/* for VxWorks and Windows */
#if (defined(_BYTE_ORDER) && defined(_BIG_ENDIAN) && (_BYTE_ORDER == _BIG_ENDIAN))
#define cmChangeByteOrder32(_v)    ((((_v) & 0xFF000000) >> 24) |  \
                                    (((_v) & 0x00FF0000) >> 8)  |  \
                                    (((_v) & 0x0000FF00) << 8)  |  \
                                    (((_v) & 0x000000FF) << 24)    \
                                   )
#else /* (_BYTE_ORDER == _BIG_ENDIAN) */
#define cmChangeByteOrder32(_v)    (_v)
#endif /* (_BYTE_ORDER == _BIG_ENDIAN) */
#endif /* __linux__ */

/*====================================================================
* Substitution Table
*--------------------------------------------------------------------
* NOTES:
*   https://en.wikipedia.org/wiki/Rijndael_S-box
*====================================================================
*/
static const unsigned char AES_SBOX[] = {
    0x63, 0x7C, 0x77, 0x7B, 0xF2, 0x6B, 0x6F, 0xC5, 0x30, 0x01, 0x67, 0x2B, 0xFE, 0xD7, 0xAB, 0x76,
    0xCA, 0x82, 0xC9, 0x7D, 0xFA, 0x59, 0x47, 0xF0, 0xAD, 0xD4, 0xA2, 0xAF, 0x9C, 0xA4, 0x72, 0xC0,
    0xB7, 0xFD, 0x93, 0x26, 0x36, 0x3F, 0xF7, 0xCC, 0x34, 0xA5, 0xE5, 0xF1, 0x71, 0xD8, 0x31, 0x15,
    0x04, 0xC7, 0x23, 0xC3, 0x18, 0x96, 0x05, 0x9A, 0x07, 0x12, 0x80, 0xE2, 0xEB, 0x27, 0xB2, 0x75,
    0x09, 0x83, 0x2C, 0x1A, 0x1B, 0x6E, 0x5A, 0xA0, 0x52, 0x3B, 0xD6, 0xB3, 0x29, 0xE3, 0x2F, 0x84,
    0x53, 0xD1, 0x00, 0xED, 0x20, 0xFC, 0xB1, 0x5B, 0x6A, 0xCB, 0xBE, 0x39, 0x4A, 0x4C, 0x58, 0xCF,
    0xD0, 0xEF, 0xAA, 0xFB, 0x43, 0x4D, 0x33, 0x85, 0x45, 0xF9, 0x02, 0x7F, 0x50, 0x3C, 0x9F, 0xA8,
    0x51, 0xA3, 0x40, 0x8F, 0x92, 0x9D, 0x38, 0xF5, 0xBC, 0xB6, 0xDA, 0x21, 0x10, 0xFF, 0xF3, 0xD2,
    0xCD, 0x0C, 0x13, 0xEC, 0x5F, 0x97, 0x44, 0x17, 0xC4, 0xA7, 0x7E, 0x3D, 0x64, 0x5D, 0x19, 0x73,
    0x60, 0x81, 0x4F, 0xDC, 0x22, 0x2A, 0x90, 0x88, 0x46, 0xEE, 0xB8, 0x14, 0xDE, 0x5E, 0x0B, 0xDB,
    0xE0, 0x32, 0x3A, 0x0A, 0x49, 0x06, 0x24, 0x5C, 0xC2, 0xD3, 0xAC, 0x62, 0x91, 0x95, 0xE4, 0x79,
    0xE7, 0xC8, 0x37, 0x6D, 0x8D, 0xD5, 0x4E, 0xA9, 0x6C, 0x56, 0xF4, 0xEA, 0x65, 0x7A, 0xAE, 0x08,
    0xBA, 0x78, 0x25, 0x2E, 0x1C, 0xA6, 0xB4, 0xC6, 0xE8, 0xDD, 0x74, 0x1F, 0x4B, 0xBD, 0x8B, 0x8A,
    0x70, 0x3E, 0xB5, 0x66, 0x48, 0x03, 0xF6, 0x0E, 0x61, 0x35, 0x57, 0xB9, 0x86, 0xC1, 0x1D, 0x9E,
    0xE1, 0xF8, 0x98, 0x11, 0x69, 0xD9, 0x8E, 0x94, 0x9B, 0x1E, 0x87, 0xE9, 0xCE, 0x55, 0x28, 0xDF,
    0x8C, 0xA1, 0x89, 0x0D, 0xBF, 0xE6, 0x42, 0x68, 0x41, 0x99, 0x2D, 0x0F, 0xB0, 0x54, 0xBB, 0x16
};


/*====================================================================
* PURPOSE: S-box substitution: Word
*--------------------------------------------------------------------
* PARAMS:
*    IN/OUT    w    : pointer to data block(state); 16 bytes.
*
* RETURNS:
*    NONE
*====================================================================
*/
static void SubWord(unsigned int* w)
{
    *w = (unsigned int)(AES_SBOX[*w & 0xff]
        | AES_SBOX[*w >> 8 & 0xff] << 8
        | AES_SBOX[*w >> 16 & 0xff] << 16
        | AES_SBOX[*w >> 24 & 0xff] << 24);
}

/*====================================================================
* PURPOSE: Shift one word
*--------------------------------------------------------------------
* PARAMS:
*    IN/OUT    w    : pointer to data block(state); 1 word (4 bytes).
*
* RETURNS:
*    NONE
*====================================================================
*/
static void ShiftWord(unsigned int* w)
{
    *w = *w << 24 | *w >> 8;
}

/*====================================================================
* Key Schedule
*--------------------------------------------------------------------
* NOTES:
*   https://en.wikipedia.org/wiki/Rijndael_key_schedule
*   Only for 16 bytes key.
*====================================================================
*/
static const unsigned int AES_RCON[] = {
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1B, 0x36
};

/*====================================================================
* PURPOSE: Expand key to round key.
*--------------------------------------------------------------------
* PARAMS:
*    IN    key    : pointer to key; 16 bytes.
*    OUT    rk    : pointer to expanded key(round key); 44 words
*
* RETURNS:
*    NONE
*
* NOTES:
*    Only for 128 bit key for now.
*    https://en.wikipedia.org/wiki/File:AES-Key_Schedule_128-bit_key.svg
*====================================================================
*/
static void ExpandsKey(const unsigned char* key, unsigned int* rk)
{
    unsigned int i, r;
    unsigned int Nk = 4;        /* The number of words in the Key ( 4 words == 128bits ), 6 for 192, 8 for 256 */
    unsigned int Nr = Nk + 6;    /* The number of Round. 10 times for 128 bit key. */
    unsigned int* cur = rk;
    unsigned int* next;

    /* prepare rk[0]~rk[3] */
    for (i = 0; i < 4; i++)
    {
        *cur++ = (unsigned int)(*key) ^ (unsigned int)((*(key + 1)) << 8) ^ (unsigned int)((*(key + 2)) << 16) ^ (unsigned int)(((*(key + 3)) << 24));
        key += 4;
    }

    cur = rk;
    next = rk + 4;

    /* generate rk[4]~rk[43] */
    for (r = 0; r < Nr; r++)
    {
        /* first Row */
        *next = *(cur + 3);
        ShiftWord(next);
        SubWord(next);
        *next ^= AES_RCON[r] ^ *cur;

        *(next + 1) = (*++cur) ^ *next;    next++;
        *(next + 1) = (*++cur) ^ *next;    next++;
        *(next + 1) = (*++cur) ^ *next;    next++;
        if (r <= Nr - 1)
        {
            next++; cur++;
        }
    }
}

/*====================================================================
* PURPOSE: Increase the count in the iv
*--------------------------------------------------------------------
* PARAMS:
*    IN    iv    : pointer to initial vector; (16 bytes, last word contains count number)
*
* RETURNS:
*    NONE
*====================================================================
*/
static void IncIV(unsigned char* const iv)
{
    unsigned int c = ((unsigned int)(*(iv + 12)) << 24)
        | ((unsigned int)(*(iv + 13)) << 16)
        | ((unsigned int)(*(iv + 14)) << 8)
        | ((unsigned int)(*(iv + 15)));

    c++;
    *(iv + 12) = (unsigned char)(c >> 24);
    *(iv + 13) = (unsigned char)(c >> 16);
    *(iv + 14) = (unsigned char)(c >> 8);
    *(iv + 15) = (unsigned char)(c);
}

/*====================================================================
* PURPOSE: Initialize initial vector
*--------------------------------------------------------------------
* PARAMS:
*    IN    nonce    : pointer to nonce; (12 bytes, 96 bits)
*    OUT    iv    : pointer to vector;
*
* RETURNS:
*    NONE
*====================================================================
*/
static void InitIV(const unsigned char* nonce, unsigned char* iv)
{
    memset(iv, 0, AES_BLOCK_SIZE);
    memcpy(iv, nonce, NONCE_LEN);    /*nonce is 12bytes. */
    IncIV(iv);                /*set 1 as initial value. */
}

/*====================================================================
* PURPOSE: Plain text block to state
*--------------------------------------------------------------------
* PARAMS:
*    IN    plainText: pointer to data block; 4 words.
*    OUT    state    : pointer to round key; 4 words
*
* RETURNS:
*    NONE
*====================================================================
*/
static void PlainExState(const unsigned char* plainText, unsigned char* state)
{
    unsigned int i;

    for (i = 0; i < 4; ++i)
    {

        *state = *(plainText);
        *(state + 4) = *(plainText + 1);
        *(state + 8) = *(plainText + 2);
        *(state + 12) = *(plainText + 3);

        state++;
        plainText += 4;
    }
}

/*====================================================================
* PURPOSE: S-box substitution: state
*--------------------------------------------------------------------
* PARAMS:
*    IN/OUT    state    : pointer to data block(state); 16 bytes.
*
* RETURNS:
*    NONE
*====================================================================
*/
static void SubBytes(unsigned char* state)
{
    unsigned int i;

    for (i = 0; i < AES_BLOCK_SIZE; i++)
    {
        *state = AES_SBOX[*state];
        state++;
    }
}

/*====================================================================
* PURPOSE: Add round key.
*--------------------------------------------------------------------
* PARAMS:
*    IN/OUT    state    : pointer to data block(state); 4 words.
*    IN    rk    : pointer to round key; 4 words
*
* RETURNS:
*   NONE
*
* NOTES:
*
*====================================================================
*/
static void AddRoundKey(unsigned char* state, const unsigned char* rk)
{
    unsigned int i;

    for (i = 0; i < 4; ++i)
    {
        unsigned int uint32RK = cmChangeByteOrder32((unsigned int)(*rk) ^
            (unsigned int)((*(rk + 1)) << 8) ^
            (unsigned int)((*(rk + 2)) << 16) ^
                                                    (unsigned int)(((*(rk + 3)) << 24)));

        *(state) = (unsigned char)((*state) ^ (uint32RK & 0x000000FF));
        *(state + 4) = (unsigned char)((*(state + 4)) ^ ((uint32RK & 0x0000FF00) >> 8));
        *(state + 8) = (unsigned char)((*(state + 8)) ^ ((uint32RK & 0x00FF0000) >> 16));
        *(state + 12) = (unsigned char)((*(state + 12)) ^ ((uint32RK & 0xFF000000) >> 24));

        state++;
        rk += 4;
    }
}

/*====================================================================
* PURPOSE: Shift Rows of Block
*--------------------------------------------------------------------
* PARAMS:
*    IN/OUT    state    : pointer to data block(state); 16 bytes.
*
* RETURNS:
*    NONE
*
* NOTES:
*    First Line: do nothing.
*    Second Line: left shift 1 byte.
*    Third Line: left shift 2 bytes.
*    Forth Line: left shift 3 bytes.
*
*====================================================================
*/
static void ShiftRows(unsigned char * state)
{
    unsigned int * w = (unsigned int*)(state + 4);
    unsigned int uint32Temp = cmChangeByteOrder32(*w);

    /* Second Line */
    uint32Temp = uint32Temp >> 8  | (uint32Temp << 24);
    *w = cmChangeByteOrder32(uint32Temp);
    w++;

    /* Third Line */
    uint32Temp = cmChangeByteOrder32(*w);
    uint32Temp = uint32Temp << 16 | ((uint32Temp >> 16) & 0xFFFF);
    *w = cmChangeByteOrder32(uint32Temp);
    w++;

    /* Forth Line */
    uint32Temp = cmChangeByteOrder32(*w);
    uint32Temp = uint32Temp << 8  | (uint32Temp >> 24);
    *w = cmChangeByteOrder32(uint32Temp);
}

/*====================================================================
* PURPOSE: Multiply by 2
*--------------------------------------------------------------------
* PARAMS:
*    IN    val    : pointer to input data
*    OUT        : result
*
* RETURNS:
*   NONE
*
* NOTES:
*    data << 1; XOR 0x1b if MSB is 1.
*
*====================================================================
*/
static unsigned char MulBy2(unsigned char* val)
{
    unsigned char tmp = *val;
    unsigned int msb = tmp & 0x80;

    tmp = (unsigned char)(tmp << 0x01);
    if (msb)
    {
        tmp ^= 0x1b;
    }
    return tmp;
}

/*====================================================================
* PURPOSE: Mix Columns with circular MDS matrix
*--------------------------------------------------------------------
* PARAMS:
*    IN/OUT    state    : pointer to data block(state); 4 words.
*    IN    rk    : pointer to round key; 4 words
*
* RETURNS:
*    NONE
*
* Matrix:
*
*   | 02 03 01 01 |
*   | 01 02 03 01 |
*   | 01 01 02 03 |
*   | 03 01 01 02 |
*
*====================================================================
*/
static void MixCol(unsigned char* state)
{
    unsigned int i;
    unsigned char org1, org2, org3, org4;
    unsigned char mul2_1, mul2_2, mul2_3, mul2_4;

    for (i = 0; i < 4; ++i)
    {
        org1 = *state;
        org2 = *(state + 4);
        org3 = *(state + 8);
        org4 = *(state + 12);

        mul2_1 = MulBy2(&org1);
        mul2_2 = MulBy2(&org2);
        mul2_3 = MulBy2(&org3);
        mul2_4 = MulBy2(&org4);

        *state = mul2_1 ^ mul2_2^ org2 ^ org3 ^ org4;
        *(state + 4) = org1 ^ mul2_2 ^ mul2_3^ org3 ^ org4;
        *(state + 8) = org1 ^ org2 ^ mul2_3 ^ mul2_4 ^ org4;
        *(state + 12) = mul2_1 ^ org1 ^ org2 ^ org3 ^ mul2_4;

        state++;
    }
}

/*====================================================================
* PURPOSE: Encrypt one state
*--------------------------------------------------------------------
* PARAMS:
*    IN    rk        : pointer to expended round key; (44 words)
*    IN    state    : pointer to data block(state); 4 words.
*    OUT    out    : pointer to encrypted data block; 4 words.
*
* RETURNS:
*    NONE
*====================================================================
*/
static void AesEncBlock(const unsigned int* rk, const unsigned char* block, unsigned char* out)
{
    unsigned int i;
    unsigned char buf[AES_BLOCK_SIZE];

    PlainExState(block, buf);

    /* Initial round */
    AddRoundKey(buf, (unsigned char*)rk);        /* AddRoundKey */

    /* 9 rounds */
    for (i = 1; i < 10; ++i)
    {
        SubBytes(buf);                /* SubBytes */
        ShiftRows(buf);                /* ShifRows */
        MixCol(buf);                /* MixColumns */
        AddRoundKey(buf, (unsigned char*)(rk + i * 4));    /* AddRoundKey */
    }

    /* Final round */
    SubBytes(buf);                /* SubBytes */
    ShiftRows(buf);                /* ShifRows */
    AddRoundKey(buf, (unsigned char*)(rk + 40));    /* AddRoundKey */

    PlainExState(buf, out);
}

/*====================================================================
* PURPOSE: XOR two blocks
*--------------------------------------------------------------------
* PARAMS:
*    IN    in    : pointer to in data 1; (16 bytes)
*    IN    out    : pointer to in data 2; (16 bytes)
*    OUT    out    : pointer to result data; (16 bytes)
*
* RETURNS:
*    NONE
*====================================================================
*/
static void XorBlocks(const unsigned char* in, unsigned char* out)
{
    unsigned int *i = (unsigned int *)in;
    unsigned int *o = (unsigned int *)out;

    *o++ ^= *i++;
    *o++ ^= *i++;
    *o++ ^= *i++;
    *o++ ^= *i++;
}

/*====================================================================
* PURPOSE: Encrypt plain text.(AES Block Encryption)
*--------------------------------------------------------------------
* PARAMS:
*    IN    key    : pointer to encryption key; (16 bytes).
*    IN    nonce    : pointer to nonce; 98 bites.
*    IN    inText    : pointer to input text.
*    IN    textLen    : input text length.
*    OUT    outText    : pointer to output text(encrypted or decrypted)
*
* RETURNS:
*    NONE
*====================================================================
*/
void aesGcmEnc(const unsigned char* key, const unsigned char* nonce, const unsigned char* inText, const unsigned int textLen, unsigned char* outText)
{
    unsigned int i;
    unsigned int roundKey[EXTKEY_LEN]; /* Expanded round key buffer */
    unsigned int blocks = textLen / AES_BLOCK_SIZE;    /* number of blocks. */
    unsigned int left = textLen % AES_BLOCK_SIZE;    /* bytes of last block. */
    unsigned char buffer[AES_BLOCK_SIZE] = { 0 };
    unsigned char iv[AES_BLOCK_SIZE] = { 0 };
    unsigned char* out;
    int flag = 0;

    /* Encrypt in the input memory space. */
    if (inText == outText)
    {
        flag = 1;
        out = buffer;
    }
    else
    { /* Encrypt in output memory space */
        out = outText;
    }

    /* Prepare round key */
    ExpandsKey(key, roundKey);

    /* Encryption */
    InitIV(nonce, iv);
    for (i = 0; i < blocks; ++i)
    {
        IncIV(iv);
        AesEncBlock(roundKey, iv, out);
        if (flag)
        {
            XorBlocks(out, outText);
            outText += AES_BLOCK_SIZE;
        }
        else
        {
            XorBlocks(inText, out);
            out += AES_BLOCK_SIZE;
        }
        inText += AES_BLOCK_SIZE;
    }
    if (left != 0)
    {
        IncIV(iv);
        AesEncBlock(roundKey, iv, buffer);
        for (i = 0; i < left; ++i)
        {
            if (flag)
            {
                *outText++ ^= *(buffer + i);
            }
            else
            {
                *out++ = *inText++ ^ *(buffer + i);
            }
        }
    }
}
